#include "codec.h"
#include <Wire.h>
#include <Arduino.h>

//AC101 CODEC

inputMode = IM_LR;

/*
//USE THESE VALUES IN SET_SETTINGS AND SET_CODEC

//CONTROL INPUT
#define P1_PIN  39
#define P2_PIN  36
#define P3_PIN  34
#define P4_PIN  14
#define P5_PIN  13
#define P6_PIN  4

//ROTARY ENCODER
#define RE_BUTTON_PIN 19
#define RE_PHASE0_PIN 23
#define RE_PHASE1_PIN 18

//FOOT SW PIN SETUP
#define FS_PIN 5

//LED INDICATOR PIN SETUP
#define MAINLED_PIN 2
#define AUXLED_PIN 15

//OLED Display PIN SETUP
#define SCK_PIN 22
#define SDA_PIN 21

//Digital input Generic Assignment
#define D1_PIN_AC101	FS_PIN
#define D2_PIN_AC101	RE_BUTTON_PIN
#define D3_PIN_AC101	RE_PHASE0_PIN
#define D4_PIN_AC101	RE_PHASE1_PIN

//ESP32-A1S-AC101 PIN SETUP
#define I2S_NUM         (0)
#define I2S_MCLK        (GPI_NUM_0)
#define I2S_BCK_IO      (GPIO_NUM_27)
#define I2S_WS_IO       (GPIO_NUM_26)
#define I2S_DO_IO       (GPIO_NUM_25)
#define I2S_DI_IO       (GPIO_NUM_35)

#define AC101_SDA		(GPIO_NUM_33)
#define AC101_SCK		(GPIO_NUM_32)
#define AC101_ADDR 		0x1A

//audio processing frame length in samples (L+R) 64 samples (32R+32L) 256 Bytes
#define FRAMELENGTH    64
//sample count per channel for each frame (32)re
#define SAMPLECOUNT   FRAMELENGTH/2
//channel count inside a frame (always stereo = 2)
#define CHANNELCOUNT  2
//frame size in bytes
#define FRAMESIZE   FRAMELENGTH*4
//audio processing priority
#define AUDIO_PROCESS_PRIORITY  10

//dma buffer length 32 bytes (8 samples: 4L+4R)
#define DMABUFFERLENGTH 32

//dma buffer count 20 (640 Bytes: 160 samples: 80L+80R) 
//#define DMABUFFERCOUNT  20
//dma buffer count 2 (for 2 ms measured latency)
#define DMABUFFERCOUNT  2

void codecsetup_task(void* arg)
{	
	if(_deviceType==DT_ESP32_A1S_AC101)
	{
		codecBusInit(AC101_SDA, AC101_SCK, 400000);
		_codecAddress = AC101_ADDR;
		
		_acodec = new AC101Codec();
		_acodec->setInputMode(_module->inputMode);
		bool res = _acodec->init(_codecAddress);
		_acodec->muteLeftAdcIn = &_muteLeftAdcIn;
		_acodec->muteRightAdcIn = &_muteRightAdcIn;
		_outCorrectionGain = _acodec->outCorrectionGain;
	}
	
*/

//AC101 REGISTERS
////////////////////////////////////////////////////////////////////////
#define AC101_ADDR			0x1A
#define CHIP_AUDIO_RS		0x00
#define PLL_CTRL1			0x01
#define PLL_CTRL2			0x02
#define SYSCLK_CTRL			0x03
#define MOD_CLK_ENA			0x04
#define MOD_RST_CTRL		0x05
#define I2S_SR_CTRL			0x06
#define I2S1LCK_CTRL		0x10
#define I2S1_SDOUT_CTRL		0x11
#define I2S1_SDIN_CTRL		0x12
#define I2S1_MXR_SRC		0x13
#define I2S1_VOL_CTRL1		0x14
#define I2S1_VOL_CTRL2		0x15
#define I2S1_VOL_CTRL3		0x16
#define I2S1_VOL_CTRL4		0x17
#define I2S1_MXR_GAIN		0x18
#define ADC_DIG_CTRL		0x40
#define ADC_VOL_CTRL		0x41
#define HMIC_CTRL1			0x44
#define HMIC_CTRL2			0x45
#define HMIC_STATUS			0x46
#define DAC_DIG_CTRL		0x48
#define DAC_VOL_CTRL		0x49
#define DAC_MXR_SRC			0x4C
#define DAC_MXR_GAIN		0x4D
#define ADC_APC_CTRL		0x50
#define ADC_SRC				0x51
#define ADC_SRCBST_CTRL		0x52
#define OMIXER_DACA_CTRL	0x53
#define OMIXER_SR			0x54
#define OMIXER_BST1_CTRL	0x55
#define HPOUT_CTRL			0x56
#define SPKOUT_CTRL			0x58
#define AC_DAC_DAPCTRL		0xA0
#define AC_DAC_DAPHHPFC 	0xA1
#define AC_DAC_DAPLHPFC 	0xA2
#define AC_DAC_DAPLHAVC 	0xA3
#define AC_DAC_DAPLLAVC 	0xA4
#define AC_DAC_DAPRHAVC 	0xA5
#define AC_DAC_DAPRLAVC 	0xA6
#define AC_DAC_DAPHGDEC 	0xA7
#define AC_DAC_DAPLGDEC 	0xA8
#define AC_DAC_DAPHGATC 	0xA9
#define AC_DAC_DAPLGATC 	0xAA
#define AC_DAC_DAPHETHD 	0xAB
#define AC_DAC_DAPLETHD 	0xAC
#define AC_DAC_DAPHGKPA 	0xAD
#define AC_DAC_DAPLGKPA 	0xAE
#define AC_DAC_DAPHGOPA 	0xAF
#define AC_DAC_DAPLGOPA 	0xB0
#define AC_DAC_DAPOPT   	0xB1
#define DAC_DAP_ENA     	0xB5

#define ARRAY_SIZE(x)  (sizeof(x)/sizeof(x[0]))
#define LEFT_MIC1_ENABIT		6
#define LEFT_LINELEFT_ENABIT	3
#define LEFT_LINEDIFF_ENABIT	4
#define RIGHT_MIC1_ENABIT		13
#define RIGHT_LINERIGHT_ENABIT	10
#define RIGHT_LINEDIFF_ENABIT	11

enum 
{
		SAMPLE_RATE_8000	= 0x0000,
		SAMPLE_RATE_11052	= 0x1000,
		SAMPLE_RATE_12000	= 0x2000,
		SAMPLE_RATE_16000	= 0x3000,
		SAMPLE_RATE_22050	= 0x4000,
		SAMPLE_RATE_24000	= 0x5000,
		SAMPLE_RATE_32000	= 0x6000,
		SAMPLE_RATE_44100	= 0x7000,
		SAMPLE_RATE_48000	= 0x8000,
		SAMPLE_RATE_96000	= 0x9000,
		SAMPLE_RATE_192000	= 0xa000,
};

enum { MODE_MASTER = 0x00,	MODE_SLAVE = 0x01 };

enum 
{
	WORD_SIZE_8_BITS	= 0x00,
	WORD_SIZE_16_BITS	= 0x01,
	WORD_SIZE_20_BITS	= 0x02,
	WORD_SIZE_24_BITS	= 0x03,
};

enum 
{
	DATA_FORMAT_I2S		= 0x00,
	DATA_FORMAT_LEFT	= 0x01,
	DATA_FORMAT_RIGHT	= 0x02,
	DATA_FORMAT_DSP		= 0x03,
};

enum 
{
	BCLK_DIV_1			= 0x0,
	BCLK_DIV_2			= 0x1,
	BCLK_DIV_4			= 0x2,
	BCLK_DIV_6			= 0x3,
	BCLK_DIV_8			= 0x4,
	BCLK_DIV_12			= 0x5,
	BCLK_DIV_16			= 0x6,
	BCLK_DIV_24			= 0x7,
	BCLK_DIV_32			= 0x8,
	BCLK_DIV_48			= 0x9,
	BCLK_DIV_64			= 0xa,
	BCLK_DIV_96			= 0xb,
	BCLK_DIV_128		= 0xc,
	BCLK_DIV_192		= 0xd,
};

enum 
{
	LRCK_DIV_16			= 0x0,
	LRCK_DIV_32			= 0x1,
	LRCK_DIV_64			= 0x2,
	LRCK_DIV_128		= 0x3,
	LRCK_DIV_256		= 0x4,
};




///////////////////////////////////////////////////////////////////////
//AC101 codec class


//Initialize the I2C bus at specific pins and clock frequency
	Codec::Codec(uint8_t _sda, uint8_t _scl, uint32_t _speed)
{
	_pinsda = _sda;
	_pinscl = _scl;
	_i2cspeed = _speed;
	i2c.begin(_sda, _scl, _speed);
}

	Codec::~Codec() { i2c.~TwoWire(); }

bool Codec::writeReg(uint8_t reg, uint16_t val)
{
	i2c.beginTransmission(AC101_ADDR);
	i2c.write(reg);
	i2c.write(uint8_t((val >> 8) & 0xff));
	i2c.write(uint8_t(val & 0xff));
	return (0 == i2c.endTransmission(true));
}

uint16_t Codec::readReg(uint8_t reg)
{
	i2c.beginTransmission(AC101_ADDR);
	i2c.write(reg);
	i2c.endTransmission(false);
	uint16_t val = 0u;
	if (2 == i2c.requestFrom(uint16_t(AC101_ADDR), uint8_t(2), true))
	{
		val = uint16_t(i2c.read() << 8) + uint16_t(i2c.read());
	}
	i2c.endTransmission(false);
	return val;
}; 

bool Codec::setI2sClock(uint16_t bitClockDiv, uint16_t bitClockInv, uint16_t lrClockDiv, uint16_t lrClockInv)
{
	uint16_t val = readReg(I2S1LCK_CTRL);
	val &= ~0x7FC0;
	val |= (bitClockInv ? 1 : 0) << 14;
	val |= bitClockDiv << 9;
	val |= (lrClockInv ? 1 : 0) << 13;
	val |= lrClockDiv << 6;
	return writeReg(I2S1LCK_CTRL, val);
}

bool Codec::setI2sFormat(uint16_t format)
{
	uint16_t val = readReg(I2S1LCK_CTRL);
	val &= ~0x000C;
	val |= uint16_t(format) << 2;
	return writeReg(I2S1LCK_CTRL, val);
}

bool Codec::setI2sMode(uint16_t mode)
{
	uint16_t val = readReg(I2S1LCK_CTRL);
	val &= ~0x8000;
	val |= uint16_t(mode) << 15;
	return writeReg(I2S1LCK_CTRL, val);
}

bool Codec::setI2sSampleRate(uint16_t rate)
{
	return writeReg(I2S_SR_CTRL, rate);
}

bool Codec::setI2sWordSize(uint16_t size)
{
	uint16_t val = readReg(I2S1LCK_CTRL);
	val &= ~0x0030;
	val |= uint16_t(size) << 4;
	return writeReg(I2S1LCK_CTRL, val);
}

bool Codec::omixerLeftLineLeft(bool select)
{
  uint16_t val = readReg(OMIXER_SR);
  if(select)
    val |= (uint16_t)1<<3;
  else val &= ~((uint16_t)1<<3);
  return writeReg(OMIXER_SR, val);
}

bool Codec::omixerLeftDacLeft(bool select)
{
  uint16_t val = readReg(OMIXER_SR);
  if(select)
    val |= (uint16_t)1<<1;
  else val &= ~((uint16_t)1<<1);
  return writeReg(OMIXER_SR, val);
}

bool Codec::omixerLeftMic1(bool select)
{
  uint16_t val = readReg(OMIXER_SR);
  if(select)
    val |= (uint16_t)1<<6;
  else val &= ~((uint16_t)1<<6);
  return writeReg(OMIXER_SR, val);
}

bool Codec::omixerRightLineRight(bool select)
{
  uint16_t val = readReg(OMIXER_SR);
  if(select)
    val |= (uint16_t)1<<10;
  else val &= ~((uint16_t)1<<10);
  return writeReg(OMIXER_SR, val);
}

bool Codec::omixerRightDacRight(bool select)
{
  uint16_t val = readReg(OMIXER_SR);
  if(select)
    val |= (uint16_t)1<<8;
  else val &= ~((uint16_t)1<<8);
  return writeReg(OMIXER_SR, val);
}

bool Codec::omixerRightMic1(bool select)
{
  uint16_t val = readReg(OMIXER_SR);
  if(select)
    val |= (uint16_t)1<<13;
  else val &= ~((uint16_t)1<<13);
  return writeReg(OMIXER_SR, val);
}

bool Codec::leftMic1(bool select)
{
	uint16_t val = readReg(ADC_SRC);
	if(select)
		val |= (uint16_t)1<<LEFT_MIC1_ENABIT;
	else val &= ~((uint16_t)1<<LEFT_MIC1_ENABIT);
	return writeReg(ADC_SRC, val);
}
 
bool Codec::rightMic1(bool select)
{
	uint16_t val = readReg(ADC_SRC);
	if(select)
		val |= (uint16_t)1<<RIGHT_MIC1_ENABIT;
	else val &= ~((uint16_t)1<<RIGHT_MIC1_ENABIT);
	return writeReg(ADC_SRC, val);
}

bool Codec::leftLineLeft(bool select)
{
	uint16_t val = readReg(ADC_SRC);
	if(select)
		val |= (uint16_t)1<<LEFT_LINELEFT_ENABIT;
	else val &= ~((uint16_t)1<<LEFT_LINELEFT_ENABIT);
	return writeReg(ADC_SRC, val);
}

bool Codec::rightLineRight(bool select)
{
	uint16_t val = readReg(ADC_SRC);
	if(select)
		val |= (uint16_t)1<<RIGHT_LINERIGHT_ENABIT;
	else val &= ~((uint16_t)1<<RIGHT_LINERIGHT_ENABIT);
	return writeReg(ADC_SRC, val);
}

bool Codec::leftLineDiff(bool select)
{
	uint16_t val = readReg(ADC_SRC);
	if(select)
		val |= (uint16_t)1<<LEFT_LINEDIFF_ENABIT;
	else val &= ~((uint16_t)1<<LEFT_LINEDIFF_ENABIT);
	return writeReg(ADC_SRC, val);
}

bool Codec::rightLineDiff(bool select)
{
	uint16_t val = readReg(ADC_SRC);
	if(select)
		val |= (uint16_t)1<<RIGHT_LINEDIFF_ENABIT;
	else val &= ~((uint16_t)1<<RIGHT_LINEDIFF_ENABIT);
	return writeReg(ADC_SRC, val);
}


bool Codec::init(int address)
{
	outCorrectionGain = 1;
	AC101_ADDR = address;
	bool ok = true;
	ok &= writeReg(CHIP_AUDIO_RS, 0x123);
	delay(100);
	ok &= 0x0101 == readReg(CHIP_AUDIO_RS);
	ok &= writeReg(SPKOUT_CTRL, 0xe880);

	// Enable the PLL from BCLK source at 44100 sample/s (BCLK = 64.fs = 2.8224M)
	// (FOUT=22.5792M, FIN=2.8224M, M=1, N=24, K=1, see AC101 codec datasheet)
	// PLL_CTRL1 = 0000 0001 0100 1111 (0x14f)
	// PLL_CTRL2 = 1000 0001 1000 0000 (0x8180)
	ok &= writeReg(PLL_CTRL1, 0x14f);
	ok &= writeReg(PLL_CTRL2, 0x8180);
	ok &= writeReg(SYSCLK_CTRL, 0xab08); //set the source from BCLK

	ok &= writeReg(MOD_CLK_ENA, 0x800c);
	ok &= writeReg(MOD_RST_CTRL, 0x800c);
    ok &= setI2sSampleRate(SAMPLE_RATE_44100);

	ok &= setI2sClock(BCLK_DIV_8, false, LRCK_DIV_64, true);
	ok &= setI2sMode(MODE_SLAVE);
	ok &= setI2sWordSize(WORD_SIZE_24_BITS);
	ok &= setI2sFormat(DATA_FORMAT_I2S);

	// AIF config
	ok &= writeReg(I2S1_SDOUT_CTRL, 0xc000);
	ok &= writeReg(I2S1_SDIN_CTRL, 0xc000);
	ok &= writeReg(I2S1_MXR_SRC, 0x2200);

	ok &= writeReg(ADC_SRCBST_CTRL, 0xccc4); //enable mic1 and mic2 with boost
	ok &= writeReg(ADC_SRC, 0x0);	//mute all source
	ok &= writeReg(ADC_DIG_CTRL, 0x8000); //enable adc
	ok &= writeReg(ADC_APC_CTRL, 0xbbc6); //enable adc left, adc right, mic bias

	// Path Configuration
	ok &= writeReg(DAC_MXR_SRC, 0xcc00);
	ok &= writeReg(DAC_DIG_CTRL, 0x8000); //enable dac
	ok &= writeReg(OMIXER_SR, 0x0102); //select only from dac
	ok &= writeReg(OMIXER_DACA_CTRL, 0xf080); //enabl ldac, rdac, lmixer, rmixer, 
	
	ok &= writeReg(ADC_DIG_CTRL, 0x8000); //enable adc
	ok &= writeReg(MOD_CLK_ENA,  0x800c);
	ok &= writeReg(MOD_RST_CTRL, 0x800c);

	// Eenable Output mixer and DAC
	ok &= writeReg(OMIXER_DACA_CTRL, 0xff80);
 
	// Enable Speaker output
	ok &= writeReg(SPKOUT_CTRL, 0xeabd);
	delay(10);
	// set the volume at 0dB
	ok &= setOutVol(31);
	
	//setup the input selection
	if(getInputMode() == 0) //mode IM_LR
    {
		leftLineLeft(true);
		rightLineRight(true);
    }
    else //mode IM_LMIC
    {
		leftLineLeft(true);
		rightMic1(true);
    }

	return ok;
}
	
//get and set the output level (analog gain)
//vol = 0-31
bool Codec::setOutputVolume(int vol)
{
	if (vol > 31) vol = 31;
	uint16_t val = readReg(SPKOUT_CTRL);
	val &= ~31;
	val |= vol;
	return writeReg(SPKOUT_CTRL, val);
}

int Codec::getOutputVolume()
{
	return (readReg(SPKOUT_CTRL) & 31);
}
	
//get and set microphone gain (0:0dB,1-7:30dB-48dB)
uint8_t Codec::getMicGain()
{
	uint16_t val = readReg(ADC_SRCBST_CTRL);
	return (uint8_t)(val >> 12) & 7;
}

bool Codec::setMicGain(uint8_t gain)
{
	if(gain > 7) gain = 7;
	uint16_t val = readReg(ADC_SRCBST_CTRL);
	val &=  ~(7 << 12);
	val |= gain << 12;
	return writeReg(ADC_SRCBST_CTRL, val);
}
	
//bypassed the analog input to the output, disconnect the digital i/o 
bool Codec::analogBypass(bool bypass, BYPASS_MODE bm)
{
	//connect or disconnect the DAC output to output mixer
    omixerRightDacRight(!bypass);
    omixerLeftDacLeft(!bypass);

    //connect or disconnect the ADC input
    if(getInputMode() == 0) //mode IM_LR
    {
		if((bm==BM_LR)||(bm==BM_L))
			leftLineLeft(!bypass);
		if((bm==BM_LR)||(bm==BM_R))
			rightLineRight(!bypass);
    }
    else //mode IM_LMIC
    {
		if((bm==BM_LR)||(bm==BM_L))
			leftLineLeft(!bypass);
		if((bm==BM_LR)||(bm==BM_R))
			rightMic1(!bypass);
    }

    //connect/disconnect the line/mic input to output mixer
    if(getInputMode() == 0) //mode = IM_LR
    {      
		if((bm==BM_LR)||(bm==BM_L))
			omixerLeftLineLeft(bypass);
		if((bm==BM_LR)||(bm==BM_R))
			omixerRightLineRight(bypass);
    }
    else //inputMode = IM_LMIC
    {
		if((bm==BM_LR)||(bm==BM_L))
			omixerLeftLineLeft(bypass);
		if((bm==BM_LR)||(bm==BM_R))
			omixerRightMic1(bypass);
    }
    return true;
}

//bypassed the analog input to the output, disconnect the digital input, preserve the digital output connection
bool Codec::analogSoftBypass(bool bypass, BYPASS_MODE bm)
{
	//always connect the DAC output to output mixer
    omixerRightDacRight(true);
    omixerLeftDacLeft(true);

    //connect or disconnect the ADC input
    if(getInputMode() == 0) //input mode = IM_LR
    {
		if((bm==BM_LR)||(bm==BM_L))
			leftLineLeft(!bypass);
		if((bm==BM_LR)||(bm==BM_R))
			rightLineRight(!bypass);
    }
    else //input mode = IM_LMIC
    {
		if((bm==BM_LR)||(bm==BM_L))
			leftLineLeft(!bypass);
		if((bm==BM_LR)||(bm==BM_R))
			rightMic1(!bypass);
    }

    //connect/disconnect the line/mic input to output mixer
    if(getInputMode() == 0) //mode = IM_LR
    {      
		if((bm==BM_LR)||(bm==BM_L))
			omixerLeftLineLeft(bypass);
		if((bm==BM_LR)||(bm==BM_R))
			omixerRightLineRight(bypass);
    }
    else //inputMode = IM_LMIC
    {
		if((bm==BM_LR)||(bm==BM_L))
			omixerLeftLineLeft(bypass);
		if((bm==BM_LR)||(bm==BM_R))
			omixerRightMic1(bypass);
    }
    return true;
}

//